package com.common.demo

import android.annotation.SuppressLint
import android.app.Activity
import android.bluetooth.BluetoothAdapter
import android.bluetooth.BluetoothDevice
import android.content.BroadcastReceiver
import android.content.Intent
import android.content.IntentFilter
import android.os.Bundle
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import com.common.demo.ui.theme.AndroidCommonTheme

class BluetoothScanActivity : AppCompatActivity() {

    private val bluetoothAdapter: BluetoothAdapter? by lazy {
        BluetoothAdapter.getDefaultAdapter()
    }
    private var receiver: BroadcastReceiver? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        title = "BluetoothScan"
        if (bluetoothAdapter == null) {
            Toast.makeText(this, getString(R.string.unsupported_bluetooth), Toast.LENGTH_SHORT).show()
            finish()
            return
        }

        setContent {
            AndroidCommonTheme {
                bluetoothDeviceList(
                    devices = scanBluetoothDevices(),
                    onDeviceSelected = { mac ->
                        val resultIntent = Intent().apply {
                            putExtra("mac_address", mac)
                        }
                        setResult(Activity.RESULT_OK, resultIntent)
                        finish()
                    }
                )
            }
        }
    }
    @SuppressLint("MissingPermission")
    @Composable
    private fun scanBluetoothDevices(): List<BluetoothDevice> {
        val scannedDevices = remember { mutableStateListOf<BluetoothDevice>() }

        LaunchedEffect(Unit) {
            if (!bluetoothAdapter!!.isEnabled) {
                bluetoothAdapter!!.enable()
            }
            scannedDevices.clear()
            receiver = object : BroadcastReceiver() {
                override fun onReceive(context: android.content.Context, intent: Intent) {
                    val action = intent.action
                    if (action == BluetoothDevice.ACTION_FOUND) {
                        val device: BluetoothDevice? =
                            intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE)
                        device?.let {
                            if (device.name == null || device.name.isEmpty()) {
                                return
                            }
                            if (device.getBluetoothClass().majorDeviceClass == 1536) {//判断是不是spp协议的蓝牙
                                if (!scannedDevices.any { it.address == device.address }) {
                                    scannedDevices.add(device)
                                }
                            }
                        }
                    }
                }
            }

            val filter = android.content.IntentFilter(BluetoothDevice.ACTION_FOUND)
            registerReceiver(receiver, filter)

            bluetoothAdapter?.startDiscovery()

            // Stop scan after 12s
            kotlinx.coroutines.delay(12000)
            bluetoothAdapter?.cancelDiscovery()

        }

        return scannedDevices
    }

    @SuppressLint("MissingPermission")
    @Composable
    private fun bluetoothDeviceList(
        devices: List<BluetoothDevice>,
        onDeviceSelected: (String) -> Unit
    ) {
        Scaffold(
        ) { padding ->
            LazyColumn(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(padding)
            ) {
                items(devices) { device ->
                    Column(
                        modifier = Modifier
                            .fillMaxWidth()
                            .clickable { onDeviceSelected(device.address) }
                            .padding(16.dp)
                    ) {
                        Text("${device.name}")
                        Text("${device.address}")
                    }
                    HorizontalDivider()
                }
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        if (receiver != null){
            unregisterReceiver(receiver)
        }
    }
}
