package com.common.demo

import android.content.DialogInterface
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.net.Uri
import android.os.Bundle
import android.view.View
import android.widget.EditText
import android.widget.RadioButton
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.common.demo.ui.theme.AndroidCommonTheme
import com.common.demo.utils.AlertDialogUtil
import com.common.demo.utils.ConnectionHolder.baseConnection
import com.common.demo.utils.FileUtils
import com.common.demo.utils.GlideEngine
import com.common.demo.utils.Utility
import com.common.sdk.command.common.PrinterCommonHelper
import com.common.sdk.command.cpcl.PrinterCPCLHelper
import com.common.sdk.constant.CPCLAlignEnum
import com.common.sdk.constant.CPCLBarcodeEnum
import com.common.sdk.constant.CPCLTextEnum
import com.common.sdk.constant.ImageTypeEnum
import com.common.sdk.data.BarcodeUnderText
import com.luck.picture.lib.basic.PictureSelector
import com.luck.picture.lib.config.SelectMimeType
import com.luck.picture.lib.entity.LocalMedia
import com.luck.picture.lib.interfaces.OnResultCallbackListener
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.File


class CPCLActivity: AppCompatActivity() {
    private var openDocumentLauncher: ActivityResultLauncher<Array<String>>? = null
    val cpclHelper = baseConnection?.let {PrinterCPCLHelper(it)}
    val commandHelper = baseConnection?.let {PrinterCommonHelper(it)}
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        title = "CPCL"
        setContent {
            AndroidCommonTheme {
                MainScreen (onNavigate = { target ->
                    when(target){
                        getString(R.string.sample_print_not_form) ->{
                            samplePrint(cpclHelper, false)
                        }
                        getString(R.string.sample_print_form) ->{
                            samplePrint(cpclHelper, true)
                        }
                        getString(R.string.print_image) ->{
                            openImageFile(cpclHelper)
                        }
                        getString(R.string.printer_status) ->{
                            getPrinterStatus(cpclHelper)
                        }
                        getString(R.string.print_pdf_file) ->{
                            openPDFFile()
                        }
                        getString(R.string.print_self) ->{
                            cpclHelper?.printSelf()
                        }
                        getString(R.string.print_result) ->{
                            printResult()
                        }
                        getString(R.string.print_result_no_id) ->{
                            printResultNullID()
                        }
                    }
                })
            }
        }
        openDocumentLauncher = registerForActivityResult<Array<String>, Uri>(
            ActivityResultContracts.OpenDocument()
        ) { uri: Uri? ->
            if (uri != null) {
                val file: File = Utility.uriToFileApiQ(uri, baseContext)
                if (file != null && file.exists()){
                    val bitmaps: List<Bitmap> = Utility.pdfToBitmap(baseContext, file, "1", 576)
                    if (bitmaps != null && bitmaps.isNotEmpty()){
                        printBitmap(bitmaps[0])
                    }
                }
            } else {
            }
        }
    }

    private fun printResultNullID() {
        val progressDialog = AlertDialogUtil.progressDialog(this, "print", "...")
        CoroutineScope(Dispatchers.IO).launch {
            commandHelper?.setRePrintSwitch(true)
            cpclHelper?.cleanReadBuffer()
            samplePrint(cpclHelper, false)
            val printResult = cpclHelper?.getPrintResultNullID(5000)
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                val resultStr = if (printResult == -1) {
                    "${getString(R.string.get_print_result_timeout)}"
                }else if(printResult == 0){
                    "${getString(R.string.print_normal)}"
                }else{
                    val str = when (printResult) {
                        1 -> {
                            getString(R.string.activity_Statues_nopage)
                        }
                        2 -> {
                            getString(R.string.activity_Statues_open)
                        }
                        else -> {
                            getString(R.string.activity_Statues_error)
                        }
                    }
                    "${getString(R.string.print_error)} $str"
                }
                AlertDialogUtil.sureDialog(this@CPCLActivity, "print result", resultStr, false, null)
            }
        }
    }

    private fun printResult() {
        val progressDialog = AlertDialogUtil.progressDialog(this, "print", "...")
        CoroutineScope(Dispatchers.IO).launch {
            commandHelper?.setRePrintSwitch(true)
            cpclHelper?.setTaskID(1)
            cpclHelper?.cleanReadBuffer()
            samplePrint(cpclHelper, false)
            val printResult = cpclHelper?.getPrintResult(5000)
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                val resultStr = if (printResult?.status == -1) {
                    "${getString(R.string.get_print_result_timeout)}"
                }else if(printResult?.status == 0){
                    "id: ${printResult?.id} ${getString(R.string.print_normal)}"
                }else{
                    val str = when (printResult?.status) {
                        1 -> {
                            getString(R.string.activity_Statues_nopage)
                        }
                        2 -> {
                            getString(R.string.activity_Statues_open)
                        }
                        else -> {
                            getString(R.string.activity_Statues_error)
                        }
                    }
                    "id: ${printResult?.id} ${getString(R.string.print_error)} $str"
                }
                AlertDialogUtil.sureDialog(this@CPCLActivity, "print result", resultStr, false, null)
            }
        }
    }

    private fun printBitmap(bitmap: Bitmap) {
        val alertDialogUtil = AlertDialogUtil(this)
        val builder = alertDialogUtil.setAlertDialog()
        val view: View = alertDialogUtil.setViewID(R.layout.item_print)
        val rbInch3 = view.findViewById<RadioButton>(R.id.rb_inch3)
        val rbInch2 = view.findViewById<RadioButton>(R.id.rb_inch2)
        val rbInch4 = view.findViewById<RadioButton>(R.id.rb_inch4)
        val rbZero = view.findViewById<RadioButton>(R.id.rb_zero)
        val rbShake = view.findViewById<RadioButton>(R.id.rb_shake)
        val rbBlackW = view.findViewById<RadioButton>(R.id.rb_bw)
        val edLight = view.findViewById<EditText>(R.id.ed_light)
        val rbNo = view.findViewById<RadioButton>(R.id.rb_lzo_no)
        val rbIntegrated = view.findViewById<RadioButton>(R.id.rb_integrated)
        builder.setNegativeButton(com.luck.picture.lib.R.string.ps_cancel, null)
        builder.setPositiveButton(R.string.activity_global_print
        ) { _, _ ->
            setPrintImage(
                bitmap,
                if (rbInch3.isChecked) 576 else if (rbInch2.isChecked) 384 else if (rbInch4.isChecked) 832 else 0,
                !rbZero.isChecked,
                if (rbShake.isChecked) 1 else (if (rbBlackW.isChecked) 0 else 2),
                if (rbNo.isChecked) 0 else (if (rbIntegrated.isChecked) 1 else 2), cpclHelper
            )
        }
        builder.show()
    }

    private fun openPDFFile() {
        openDocumentLauncher?.launch(arrayOf("application/pdf"))
    }

    private fun getPrinterStatus(cpclHelper: PrinterCPCLHelper?) {
        val progressDialog = AlertDialogUtil.progressDialog(this, "printer status", "...")
        CoroutineScope(Dispatchers.IO).launch {
            val printerStatus = cpclHelper?.getPrinterStatus()
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                val statusStr = if (printerStatus == 0) {
                    getString(R.string.activity_Statues_ready)
                }else if(printerStatus?.and(2) == 2){
                    getString(R.string.activity_Statues_nopage)
                }else if(printerStatus?.and( 4) == 4){
                    getString(R.string.activity_Statues_open)
                }else{
                    getString(R.string.activity_Statues_error)
                }
                AlertDialogUtil.sureDialog(this@CPCLActivity, "printer status", statusStr, false, null)
            }
        }
    }

    private fun openImageFile(cpclHelper: PrinterCPCLHelper?) {
        PictureSelector.create(this)
            .openGallery(SelectMimeType.ofImage())
            .setImageEngine(GlideEngine.createGlideEngine())
            .setMaxSelectNum(1)
            .forResult(object : OnResultCallbackListener<LocalMedia?> {
                override fun onResult(result: ArrayList<LocalMedia?>) {
                    printImage(result, cpclHelper)
                }

                override fun onCancel() {
                }
            })
    }

    private fun printImage(result: java.util.ArrayList<LocalMedia?>, cpclHelper: PrinterCPCLHelper?) {
        var path = result[0]!!.path
        if (path.contains("content://")) {
            val uri = Uri.parse(path)
            path = FileUtils.getFilePathByUri_BELOWAPI11(uri, this)
        }
        val bmp = BitmapFactory.decodeFile(path)
        if (bmp == null) {
            Toast.makeText(baseContext, "Image error", Toast.LENGTH_SHORT).show()
            return
        }
        printBitmap(bmp)
    }

    private fun setPrintImage(
        bmp: Bitmap,
        width: Int,
        isRotate: Boolean,
        type: Int,
        compressType: Int,
        cpclHelper: PrinterCPCLHelper?
    ) {
        val progressDialog = AlertDialogUtil.progressDialog(this, "print image", "printing...")
        CoroutineScope(Dispatchers.IO).launch {
            var bitmapPrint: Bitmap = bmp
            if (isRotate) bitmapPrint = Utility.Tobitmap90(bitmapPrint)
            if (width != 0) bitmapPrint = Utility.toBitmap(
                bitmapPrint,
                width,
                Utility.getHeight(width, bitmapPrint.width, bitmapPrint.height)
            )
            cpclHelper?.printAreaSize("0","200","200", bitmapPrint.height.toString(), "1")
            cpclHelper?.printBitmapCPCL(bitmapPrint,0,0, if (type == 0) ImageTypeEnum.THRESHOLD else if(type == 1) ImageTypeEnum.RASTERMONO else ImageTypeEnum.POLYMERIZATION, compressType, 0)
            cpclHelper?.print()
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
            }
        }
    }

    private fun samplePrint(cpclHelper: PrinterCPCLHelper?, isForm: Boolean) {
        cpclHelper?.printAreaSize("0", "200", "200", "1300", "1")
        cpclHelper?.align(CPCLAlignEnum.CENTER)
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "50", "5", "TEST_PAGE")
        cpclHelper?.align(CPCLAlignEnum.LEFT)
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "66", "CODE128")
        cpclHelper?.barcode(CPCLBarcodeEnum.BARCODE, "128", "2", "1", "50", "0", "100", "123456789", BarcodeUnderText("7", "0", "5"))
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "180", "UPCA")
        cpclHelper?.barcode(CPCLBarcodeEnum.BARCODE, "UPCA", "2", "1", "50", "0", "210", "123456789012", BarcodeUnderText("7", "0", "5"))
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "310", "EAN8")
        cpclHelper?.barcode(CPCLBarcodeEnum.BARCODE, "EAN8", "2", "1", "50", "0", "340", "12345678", BarcodeUnderText("7", "0", "5"))
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "440", "CODE93")
        cpclHelper?.barcode(CPCLBarcodeEnum.BARCODE, "93", "2", "1", "50", "0", "470", "123456789", BarcodeUnderText("7", "0", "5"))
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "570", "CODE39")
        cpclHelper?.barcode(CPCLBarcodeEnum.BARCODE, "39", "2", "1", "50", "0", "600", "123456789", BarcodeUnderText("7", "0", "5"))
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "700", "QRCode")
        cpclHelper?.printQR(CPCLBarcodeEnum.BARCODE, "0", "730", "4", "6", "ABC123")
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "870", "Line")
        cpclHelper?.line("0", "910", "400", "910", "1")
        cpclHelper?.text(CPCLTextEnum.TEXT, "8", "0", "0", "950", "Box")
        cpclHelper?.box("10", "980", "400", "1200", "1")
        if (isForm) {
            cpclHelper?.form()
        }
        cpclHelper?.print()
    }
}
@Composable
fun MainScreen(onNavigate: (String) -> Unit) {
    val connectionOptions = listOf(
        stringResource(R.string.sample_print_not_form),
        stringResource(R.string.sample_print_form),
        stringResource(R.string.print_image),
        stringResource(R.string.printer_status),
        stringResource(R.string.print_pdf_file),
        stringResource(R.string.print_self),
        stringResource(R.string.print_result),
        stringResource(R.string.print_result_no_id))
    Scaffold(
    ) { padding ->
        LazyColumn(
            modifier = Modifier
                .fillMaxSize()
                .padding(padding)
        ) {
            items(connectionOptions) { option ->
                Column(
                    modifier = Modifier
                        .fillMaxWidth()
                        .clickable { onNavigate(option) }
                        .padding(16.dp)
                ) {
                    Text("${option}")
                }
                HorizontalDivider()
            }
        }
    }
}