package com.common.demo

import android.os.Bundle
import androidx.activity.compose.setContent
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.common.demo.ui.theme.AndroidCommonTheme
import com.common.demo.utils.AlertDialogUtil
import com.common.demo.utils.ConnectionHolder.baseConnection
import com.common.sdk.command.common.PrinterCommonHelper
import com.common.sdk.constant.PaperTypeEnum
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext


class CommonActivity: AppCompatActivity() {
    private val commandHelper = baseConnection?.let {
        PrinterCommonHelper(
            it
        )
    }
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        title = "Command"
        setContent {
            AndroidCommonTheme {
                MainScreenCommand (onNavigate = { target ->
                    when(target){
                        getString(R.string.get_printer_sn) ->{
                            getPrinterSN()
                        }
                        getString(R.string.get_printer_model) ->{
                            getPrinterModel()
                        }
                        getString(R.string.get_printer_version) ->{
                            getPrinterVersion()
                        }
                        getString(R.string.get_printer_paper_type) ->{
                            getPrinterPaperType()
                        }
                        getString(R.string.set_printer_paper_type) ->{
                            setPrinterPaperType()
                        }
                    }
                })
            }
        }
    }

    private fun setPrinterPaperType() {
        val builder = MaterialAlertDialogBuilder(this)
        builder.setIcon(R.drawable.img)
        val cities = resources.getStringArray(R.array.printer_paper_type)
        builder.setItems(cities) { _, which ->
            commandHelper?.setPrinterPaperType(PaperTypeEnum.values()[which])
        }
        builder.show()
    }

    private fun getPrinterPaperType() {
        val progressDialog = AlertDialogUtil.progressDialog(this, getString(R.string.get_printer_paper_type), "...")
        CoroutineScope(Dispatchers.IO).launch {
            val printerVersion = commandHelper?.printerPaperType
            val str = when(printerVersion){
                -1 ->{
                    "get time out"
                }
                PaperTypeEnum.CONTINUOUS.value ->{
                    resources.getStringArray(R.array.printer_paper_type)[0]
                }
                PaperTypeEnum.LABEL.value ->{
                    resources.getStringArray(R.array.printer_paper_type)[1]
                }
                PaperTypeEnum.TWO_FORMER_BLACK_LABEL.value ->{
                    resources.getStringArray(R.array.printer_paper_type)[2]
                }
                PaperTypeEnum.THREE_FORMER_BLACK_LABEL.value ->{
                    resources.getStringArray(R.array.printer_paper_type)[3]
                }
                PaperTypeEnum.TWO_AFTER_BLACK_LABEL.value ->{
                    resources.getStringArray(R.array.printer_paper_type)[4]
                }
                PaperTypeEnum.THREE_AFTER_BLACK_LABEL.value ->{
                    resources.getStringArray(R.array.printer_paper_type)[5]
                }
                else -> {
                    "Unknown"
                }
            }
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                AlertDialogUtil.sureDialog(this@CommonActivity, getString(R.string.get_printer_paper_type), str, false, null)
            }
        }
    }

    private fun getPrinterVersion() {
        val progressDialog = AlertDialogUtil.progressDialog(this, "Get Version", "...")
        CoroutineScope(Dispatchers.IO).launch {
            val printerVersion = commandHelper?.printerVersion
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                AlertDialogUtil.sureDialog(this@CommonActivity, "printer version", printerVersion, false, null)
            }
        }
    }

    private fun getPrinterModel() {
        val progressDialog = AlertDialogUtil.progressDialog(this, "Get Model", "...")
        CoroutineScope(Dispatchers.IO).launch {
            val printerModel = commandHelper?.printerModel
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                AlertDialogUtil.sureDialog(this@CommonActivity, "printer model", printerModel, false, null)
            }
        }
    }

    private fun getPrinterSN() {
        val progressDialog = AlertDialogUtil.progressDialog(this, "Get SN", "...")
        CoroutineScope(Dispatchers.IO).launch {
            val printerSN = commandHelper?.printerSN
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                AlertDialogUtil.sureDialog(this@CommonActivity, "printer sn", printerSN, false, null)
            }
        }
    }
}
@Composable
fun MainScreenCommand(onNavigate: (String) -> Unit) {
    val connectionOptions = listOf(
        stringResource(R.string.get_printer_sn),
        stringResource(R.string.get_printer_model),
        stringResource(R.string.get_printer_version),
        stringResource(R.string.get_printer_paper_type),
        stringResource(R.string.set_printer_paper_type)
    )
    Scaffold(
    ) { padding ->
        LazyColumn(
            modifier = Modifier
                .fillMaxSize()
                .padding(padding)
        ) {
            items(connectionOptions) { option ->
                Column(
                    modifier = Modifier
                        .fillMaxWidth()
                        .clickable { onNavigate(option) }
                        .padding(16.dp)
                ) {
                    Text(option)
                }
                HorizontalDivider()
            }
        }
    }
}