package com.common.demo

import android.annotation.SuppressLint
import android.app.Activity
import android.app.PendingIntent
import android.app.PendingIntent.FLAG_MUTABLE
import android.bluetooth.BluetoothAdapter
import android.bluetooth.BluetoothDevice
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.hardware.usb.UsbDevice
import android.hardware.usb.UsbInterface
import android.hardware.usb.UsbManager
import android.os.Build
import android.os.Bundle
import android.os.Parcelable
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.compose.setContent
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.common.demo.ui.theme.AndroidCommonTheme
import com.common.demo.utils.AlertDialogUtil
import com.common.demo.utils.ConnectionHolder
import com.common.demo.utils.PermissionHelper
import com.common.demo.utils.Utility
import com.common.sdk.PrinterHelper
import com.common.sdk.connect.BTConnection
import com.common.sdk.connect.BaseConnection
import com.common.sdk.connect.USBConnect
import com.common.sdk.connect.WifiConnect
import com.common.sdk.constant.PublicFunction
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext


class MainActivity : AppCompatActivity() {
    private val ACTION_USB_PERMISSION: String = "com.common.demo"
    private var baseConnection: BaseConnection? = null
    private var mPermissionIntent: PendingIntent? = null
    private val statusTextState = mutableStateOf("")
    private val loadingMessageState = mutableStateOf("")
    private val isLoadingState = mutableStateOf(false)
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        permission()
        initUSB()
        title = BuildConfig.VERSION_NAME
        setContent {
            statusTextState.value = (getString(R.string.pls_connect_printer))
            var statusText by remember { statusTextState }
            var isLoading by remember { isLoadingState }
            var loadingMessage by remember { loadingMessageState }
            val launcher = rememberLauncherForActivityResult(
                contract = ActivityResultContracts.StartActivityForResult()
            ) { result ->
                if (result.resultCode == Activity.RESULT_OK) {
                    val mac = result.data?.getStringExtra("mac_address")
                    if (mac != null){
                        connectBT(mac)
                    }
                }
            }
            AndroidCommonTheme {
                if (isLoading) {
                    AlertDialog(
                        onDismissRequest = {},
                        confirmButton = {},
                        title = {
                            Text(getString(R.string.please_wait))
                        },
                        text = {
                            Row(verticalAlignment = Alignment.CenterVertically) {
                                CircularProgressIndicator(modifier = Modifier.size(20.dp))
                                Spacer(modifier = Modifier.width(8.dp))
                                Text(loadingMessage)
                            }
                        }
                    )
                }
                MainScreen(
                    statusText,
                    onNavigate = { target ->
                        if (baseConnection == null || baseConnection?.isConnect() == false){
                            Toast.makeText(this, getString(R.string.pls_connect_printer), Toast.LENGTH_SHORT).show()
                            return@MainScreen
                        }
                        when (target) {
                            "CPCL" -> startActivity(Intent(this, CPCLActivity::class.java))
                            "TSPL" -> startActivity(Intent(this, TSPLActivity::class.java))
                            "Common" -> startActivity(Intent(this, CommonActivity::class.java))
                        }
                    },
                    onConnect = { type ->
                        when(type){
                            getString(R.string.bluetooth_connect) ->{
                                launcher.launch(Intent(baseContext, BluetoothScanActivity::class.java))
                            }
                            getString(R.string.wifi_connect) ->{
                                connectWifi()
                            }
                            getString(R.string.usb_connect) ->{
                                connectUSB()
                            }
                        }
                    },
                    onDisconnect = {
                        baseConnection?.disConnect()
                        statusText = getString(R.string.pls_connect_printer)
                    }
                )
            }
        }
    }

    private fun connectBT(
        mac: String?,
    ) {
        loadingMessageState.value = getString(R.string.connect_bt)
        isLoadingState.value = true
        CoroutineScope(Dispatchers.IO).launch {
            val connection = BTConnection()
            val resultCode = connection.connectBT(mac!!)
            withContext(Dispatchers.Main) {
                isLoadingState.value = false
                statusTextState.value = if (resultCode == 0) {
                    baseConnection = connection
                    ConnectionHolder.baseConnection = baseConnection
                    baseContext.getString(R.string.bluetooth_connect) + baseContext.getString(R.string.succeed)
                } else {
                    baseContext.getString(R.string.bluetooth_connect) + baseContext.getString(R.string.fail) + resultCode
                }
            }
        }
    }

    private fun connectWifi() {
        val publicFunction = PublicFunction(baseContext)
        val ip = publicFunction.readSharedPreferencesData("IP")
        AlertDialogUtil.edTextDialog(this, "WIFI", getString(R.string.pls_input_ip), ip.ifEmpty { "192.168.1.1" }
        ) { ip ->
            loadingMessageState.value = getString(R.string.wifi_connect)
            isLoadingState.value = true
            CoroutineScope(Dispatchers.IO).launch {
                val connection = WifiConnect()
                val resultCode = connection.connectWifi(ip)
                withContext(Dispatchers.Main) {
                    isLoadingState.value = false
                    statusTextState.value = if (resultCode == 0) {
                        publicFunction.writeSharedPreferencesData("IP", ip)
                        baseConnection = connection
                        ConnectionHolder.baseConnection = baseConnection
                        baseContext.getString(R.string.wifi_connect) + baseContext.getString(R.string.succeed)
                    } else {
                        baseContext.getString(R.string.wifi_connect) + baseContext.getString(R.string.fail) + resultCode
                    }
                }
            }
        }
    }

    private fun connectUSB() {
        val mUsbManager = getSystemService(USB_SERVICE) as UsbManager
        val deviceList: HashMap<String, UsbDevice> =
            mUsbManager.getDeviceList()
        val deviceIterator: Iterator<UsbDevice> =
            deviceList.values.iterator()
        var havePrinter = false
        while (deviceIterator.hasNext()) {
            val device = deviceIterator.next()
            val count: Int = device.interfaceCount
            for (i in 0 until count) {
                val into: UsbInterface = device.getInterface(i)
                if (into.interfaceClass == 7) {
                    havePrinter = true
                    mUsbManager.requestPermission(device, mPermissionIntent)
                }
            }
        }
        if (!havePrinter) {
            statusTextState.value = getString(R.string.activity_main_connect_usb_printer)
        }
    }

    @SuppressLint("UnspecifiedImmutableFlag")
    private fun initUSB() {
        val intent = Intent(ACTION_USB_PERMISSION)
        intent.setPackage(packageName)
        mPermissionIntent = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            PendingIntent.getBroadcast(baseContext, 0, intent, FLAG_MUTABLE)
        } else {
            PendingIntent.getBroadcast(baseContext, 0, intent, 0)
        }
        val filter = IntentFilter(ACTION_USB_PERMISSION)
        filter.addAction(UsbManager.ACTION_USB_DEVICE_DETACHED)
        filter.addAction(BluetoothDevice.ACTION_ACL_DISCONNECTED)//蓝牙断开监听
        filter.addAction(BluetoothAdapter.ACTION_STATE_CHANGED)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            registerReceiver(mUsbReceiver, filter, RECEIVER_EXPORTED)
        } else {
            registerReceiver(mUsbReceiver, filter)
        }
    }
    private val mUsbReceiver: BroadcastReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            try {
                val action = intent.action
                if (ACTION_USB_PERMISSION == action) {
                    synchronized(this) {
                        val device =
                            intent.getParcelableExtra<Parcelable>(UsbManager.EXTRA_DEVICE) as UsbDevice?
                        if (intent.getBooleanExtra(UsbManager.EXTRA_PERMISSION_GRANTED, false)) {
                            val usbConnect = USBConnect()
                            val connectUSB = usbConnect.connectUSB(device!!)
                            if (connectUSB == 0) {
                                statusTextState.value = getString(R.string.usb_connect) + getString(R.string.succeed)
                                baseConnection = usbConnect
                                ConnectionHolder.baseConnection = baseConnection
                            } else {
                                statusTextState.value = getString(R.string.usb_connect) + getString(R.string.fail) + connectUSB
                            }
                        } else {
                            return
                        }
                    }
                }
                if (UsbManager.ACTION_USB_DEVICE_DETACHED == action) {
                    val device =
                        intent.getParcelableExtra<Parcelable>(UsbManager.EXTRA_DEVICE) as UsbDevice?
                    if (device != null) {
                        val count: Int = device.getInterfaceCount()
                        for (i in 0 until count) {
                            val intf: UsbInterface = device.getInterface(i)
                            //Class ID 7代表打印机
                            if (intf.interfaceClass == 7) {
                            }
                        }
                    }
                }
                if (BluetoothDevice.ACTION_ACL_DISCONNECTED == action){
                    baseConnection?.disConnect()
                    statusTextState.value = getString(R.string.pls_connect_printer)
                }
                if (BluetoothAdapter.ACTION_STATE_CHANGED == action){
                    val state = intent.getIntExtra(
                        BluetoothAdapter.EXTRA_STATE,
                        BluetoothAdapter.ERROR
                    )
                    when (state) {
                        BluetoothAdapter.STATE_OFF -> {
                            if (baseConnection?.isConnect() == true) {
                                baseConnection?.disConnect()
                                statusTextState.value = getString(R.string.pls_connect_printer)
                            }
                        }
                    }
                }
            } catch (e: Exception) {
            }
        }
    }

    private fun permission() {
        val permissionHelper = PermissionHelper(
            activity = this
        ) { granted, deniedList ->
            if (granted) {
//                Toast.makeText(this, "权限已全部授权", Toast.LENGTH_SHORT).show()
            } else {
//                Toast.makeText(this, "未授权权限: $deniedList", Toast.LENGTH_SHORT).show()
            }
        }

        permissionHelper.request(
            android.Manifest.permission.BLUETOOTH_ADMIN,
            android.Manifest.permission.BLUETOOTH,
            android.Manifest.permission.BLUETOOTH_CONNECT,
            android.Manifest.permission.BLUETOOTH_SCAN,
            android.Manifest.permission.ACCESS_FINE_LOCATION,
            android.Manifest.permission.ACCESS_COARSE_LOCATION,
            android.Manifest.permission.READ_MEDIA_IMAGES,
            android.Manifest.permission.READ_MEDIA_AUDIO,
            android.Manifest.permission.READ_MEDIA_VIDEO,
            android.Manifest.permission.READ_EXTERNAL_STORAGE
        )
    }

    override fun onDestroy() {
        super.onDestroy()
        baseConnection?.disConnect()
    }
}

@Composable
fun MainScreen(statusText: String, onNavigate: (String) -> Unit, onConnect: (type: String) -> Unit,
               onDisconnect: () -> Unit) {
    val bluetooth =
        stringResource(id = R.string.bluetooth_connect)
    val wifi =
        stringResource(id = R.string.wifi_connect)
    val usb =
        stringResource(id = R.string.usb_connect)
    val connectionOptions = listOf("CPCL", "TSPL", "Common")

    Scaffold(
    ) { innerPadding ->
        Column(modifier = Modifier.fillMaxSize().padding(innerPadding)
        ) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.spacedBy(5.dp)
            ) {
                Button(onClick = {
                    onConnect(bluetooth) },
                    modifier = Modifier.weight(1f)) {
                    Box(
                        modifier = Modifier.fillMaxWidth(),
                        contentAlignment = Alignment.Center
                    ) {
                        Text(bluetooth)
                    }
                }
                Button(onClick = {
                    onConnect(wifi) },
                    modifier = Modifier.weight(1f)) {
                    Box(
                        modifier = Modifier.fillMaxWidth(),
                        contentAlignment = Alignment.Center
                    ) {
                        Text(wifi)
                    }
                }
                Button(onClick = {
                    onConnect(usb)},
                    modifier = Modifier.weight(1f)) {
                    Box(
                        modifier = Modifier.fillMaxWidth(),
                        contentAlignment = Alignment.Center
                    ) {
                        Text(usb)
                    }
                }
            }
            Spacer(modifier = Modifier.height(5.dp))
            Row(
                modifier = Modifier.fillMaxWidth(),
                verticalAlignment = Alignment.CenterVertically
            ){
                Text(
                    text = statusText,
                    style = MaterialTheme.typography.bodyLarge,
                    modifier = Modifier
                        .weight(1f)
                        .padding(end = 8.dp)
                )
                Button(onClick = { onDisconnect() }) {
                    Text(stringResource(id = R.string.disconnect))
                }
            }
            Spacer(modifier = Modifier.height(16.dp))
            LazyColumn {
                items(connectionOptions) { option ->
                    Text(
                        text = option,
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(12.dp)
                            .clickable { onNavigate(option) }
                    )
                    HorizontalDivider(thickness = 1.dp)
                }
            }
        }
    }
}