package com.common.demo

import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.net.Uri
import android.os.Bundle
import android.view.View
import android.widget.EditText
import android.widget.RadioButton
import android.widget.Toast
import androidx.activity.compose.setContent
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import com.common.demo.ui.theme.AndroidCommonTheme
import com.common.demo.utils.AlertDialogUtil
import com.common.demo.utils.ConnectionHolder.baseConnection
import com.common.demo.utils.FileUtils
import com.common.demo.utils.GlideEngine
import com.common.demo.utils.Utility
import com.common.sdk.command.tspl.PrinterTSPLHelper
import com.common.sdk.constant.ImageTypeEnum
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.luck.picture.lib.basic.PictureSelector
import com.luck.picture.lib.config.SelectMimeType
import com.luck.picture.lib.entity.LocalMedia
import com.luck.picture.lib.interfaces.OnResultCallbackListener
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.File


class TSPLActivity: AppCompatActivity() {
    private val tsplHelper = baseConnection?.let { PrinterTSPLHelper(it) }
    private var openDocumentLauncher: ActivityResultLauncher<Array<String>>? = null
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        title = "TSPL"
        setContent {
            AndroidCommonTheme {
                MainScreenTSPL (onNavigate = { target ->
                    when(target){
                        getString(R.string.print_self) ->{
                            tsplHelper?.selfTest()
                        }
                        getString(R.string.sample_print_vector) ->{
                            samplePrint(false)
                        }
                        getString(R.string.sample_print_lattice) ->{
                            samplePrint(true)
                        }
                        getString(R.string.print_image) ->{
                            openImageFile()
                        }
                        getString(R.string.printer_status) ->{
                            getPrinterStatus()
                        }
                        getString(R.string.print_block) ->{
                            printBlock()
                        }
                        getString(R.string.print_pdf_file) ->{
                            openPDFFile()
                        }
                        getString(R.string.set_black_position) ->{
                            setBlackPosition()
                        }
                    }
                })
            }
        }
        openDocumentLauncher = registerForActivityResult<Array<String>, Uri>(
            ActivityResultContracts.OpenDocument()
        ) { uri: Uri? ->
            if (uri != null) {
                val file: File = Utility.uriToFileApiQ(uri, baseContext)
                if (file != null && file.exists()){
                    val bitmaps: List<Bitmap> = Utility.pdfToBitmap(baseContext, file, "1", 576)
                    if (bitmaps != null && bitmaps.isNotEmpty()){
                        printBitmap(bitmaps[0])
                    }
                }
            } else {
            }
        }
    }

    private fun setBlackPosition() {
        val builder = MaterialAlertDialogBuilder(this)
        builder.setIcon(R.drawable.img)
        val cities = resources.getStringArray(R.array.activity_main_black)
        builder.setItems(cities) { _, which ->
            tsplHelper?.setBlackPosition(which + 1)
        }
        builder.show()
    }

    private fun openPDFFile() {
        openDocumentLauncher?.launch(arrayOf("application/pdf"))
    }

    private fun printBlock() {
        tsplHelper?.printAreaSize("100", "100")
        tsplHelper?.cls()
        tsplHelper?.printBlock(0, 0, 200, 100, 0, 0, 2, 2, 16, 2, "TestTestTestTest")
        tsplHelper?.print("1", "1")
    }

    private fun getPrinterStatus() {
        val progressDialog = AlertDialogUtil.progressDialog(this, "printer status", "...")
        CoroutineScope(Dispatchers.IO).launch {
            val printerStatus = tsplHelper?.getPrinterStatus()
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
                val statusStr = when(printerStatus){
                    PrinterTSPLHelper.STATUS_OK ->{
                        getString(R.string.activity_Statues_ready)
                    }
                    PrinterTSPLHelper.STATUS_COVER_OPENED ->{
                        getString(R.string.activity_Statues_open)
                    }
                    PrinterTSPLHelper.STATUS_NOPAPER ->{
                        getString(R.string.activity_Statues_nopage)
                    }
                    PrinterTSPLHelper.STATUS_DISCONNECT ->{
                        getString(R.string.activity_Statues_nopage)
                    }
                    PrinterTSPLHelper.STATUS_TIMEOUT ->{
                        getString(R.string.status_timeout)
                    }
                    PrinterTSPLHelper.STATUS_OVER_HEATING ->{
                        getString(R.string.status_over_heating)
                    }
                    PrinterTSPLHelper.STATUS_PRINTING ->{
                        getString(R.string.status_printing)
                    }
                    else -> {
                        getString(R.string.activity_Statues_error)
                    }
                }
                AlertDialogUtil.sureDialog(this@TSPLActivity, "printer status", statusStr, false, null)
            }
        }
    }

    private fun openImageFile() {
        PictureSelector.create(this)
            .openGallery(SelectMimeType.ofImage())
            .setImageEngine(GlideEngine.createGlideEngine())
            .setMaxSelectNum(1)
            .forResult(object : OnResultCallbackListener<LocalMedia?> {
                override fun onResult(result: ArrayList<LocalMedia?>) {
                    printImage(result)
                }

                override fun onCancel() {
                }
            })
    }

    private fun printImage(result: java.util.ArrayList<LocalMedia?>) {
        var path = result[0]!!.path
        if (path.contains("content://")) {
            val uri = Uri.parse(path)
            path = FileUtils.getFilePathByUri_BELOWAPI11(uri, this)
        }
        val bmp = BitmapFactory.decodeFile(path)
        printBitmap(bmp)
    }

    private fun printBitmap(bmp: Bitmap) {
        if (bmp == null) {
            Toast.makeText(baseContext, "Image error", Toast.LENGTH_SHORT).show()
            return
        }
        val alertDialogUtil = AlertDialogUtil(this)
        val builder = alertDialogUtil.setAlertDialog()
        val view: View = alertDialogUtil.setViewID(R.layout.item_print)
        val rbInch3 = view.findViewById<RadioButton>(R.id.rb_inch3)
        val rbInch2 = view.findViewById<RadioButton>(R.id.rb_inch2)
        val rbInch4 = view.findViewById<RadioButton>(R.id.rb_inch4)
        val rbZero = view.findViewById<RadioButton>(R.id.rb_zero)
        val rbShake = view.findViewById<RadioButton>(R.id.rb_shake)
        val rbBlackW = view.findViewById<RadioButton>(R.id.rb_bw)
        val edLight = view.findViewById<EditText>(R.id.ed_light)
        val rbNo = view.findViewById<RadioButton>(R.id.rb_lzo_no)
        val rbIntegrated = view.findViewById<RadioButton>(R.id.rb_integrated)
        builder.setNegativeButton(com.luck.picture.lib.R.string.ps_cancel, null)
        builder.setPositiveButton(
            R.string.activity_global_print
        ) { _, _ ->
            setPrintImage(
                bmp,
                if (rbInch3.isChecked) 576 else if (rbInch2.isChecked) 384 else if (rbInch4.isChecked) 832 else 0,
                !rbZero.isChecked,
                if (rbShake.isChecked) 1 else (if (rbBlackW.isChecked) 0 else 2),
                if (rbNo.isChecked) 0 else (if (rbIntegrated.isChecked) 1 else 2)
            )
        }
        builder.show()
    }

    private fun setPrintImage(
        bmp: Bitmap,
        width: Int,
        isRotate: Boolean,
        type: Int,
        compressType: Int
    ) {
        val progressDialog = AlertDialogUtil.progressDialog(this, "print image", "printing...")
        CoroutineScope(Dispatchers.IO).launch {
            var bitmapPrint: Bitmap = bmp
            if (isRotate) bitmapPrint = Utility.Tobitmap90(bitmapPrint)
            if (width != 0) bitmapPrint = Utility.toBitmap(
                bitmapPrint,
                width,
                Utility.getHeight(width, bitmapPrint.width, bitmapPrint.height)
            )
            tsplHelper?.printAreaSize("100",bitmapPrint.height.toString())
            tsplHelper?.cls()
            tsplHelper?.printImage("100", "0", bitmapPrint, true, compressType!=0, if (type == 0) ImageTypeEnum.THRESHOLD else if(type == 1) ImageTypeEnum.RASTERMONO else ImageTypeEnum.POLYMERIZATION)
            tsplHelper?.print("1","1")
            withContext(Dispatchers.Main) {
                progressDialog.dismiss()
            }
        }
    }

    private fun samplePrint(isLattice: Boolean) {
        tsplHelper?.printAreaSize("100", "100")
        tsplHelper?.cls()
        val receiptLines = resources.getStringArray(R.array.activity_main_sample_2inch_receipt)
        for (i in receiptLines.indices) {
            if (!isLattice){
                tsplHelper?.printText("50", "" + (i * 30), "9", "0", 2, receiptLines[i].trim())
            }else{
                tsplHelper?.printText("50", "" + (i * 30), "9", "0", "1","1", receiptLines[i].trim())
            }
        }
        tsplHelper?.printQRCode("10", "640", "M", "5", "M1", "0", "123ABC")
        tsplHelper?.printQRCode("200", "640", "M", "5", "M1", "0", "123ABC")
        tsplHelper?.print("1", "1")
    }
}
@Composable
fun MainScreenTSPL(onNavigate: (String) -> Unit) {
    val connectionOptions = listOf(
        stringResource(R.string.print_self),
        stringResource(R.string.sample_print_vector),
        stringResource(R.string.sample_print_lattice),
        stringResource(R.string.print_image),
        stringResource(R.string.printer_status),
        stringResource(R.string.print_block),
        stringResource(R.string.print_pdf_file),
        stringResource(R.string.set_black_position)
    )
    Scaffold(
    ) { padding ->
        LazyColumn(
            modifier = Modifier
                .fillMaxSize()
                .padding(padding)
        ) {
            items(connectionOptions) { option ->
                Column(
                    modifier = Modifier
                        .fillMaxWidth()
                        .clickable { onNavigate(option) }
                        .padding(16.dp)
                ) {
                    Text(option)
                }
                HorizontalDivider()
            }
        }
    }
}