//
//  DocumentProtocol.swift
//  PrinterDemo
//
//  Created by ios on 2022/3/11.
//  Copyright © 2022 Printer. All rights reserved.
//

import Foundation
import UIKit
import PrinterSDK
import SVProgressHUD
import SwifterBaseKit

protocol DocumentPickerProtocol {
    func selectCmdPrint(_ url: URL)
    func updatePrinterFirmware(_ url: URL)
    func selectPDFPrint(_ url: URL, _ cmdType: PTCommandType)
}

extension UIViewController : DocumentPickerProtocol {
    
    func selectCmdPrint(_ url: URL) {
        let alertControl = UIAlertController(title: "Tips".localized, message: "Please select the mode of sending data".localized, preferredStyle: UIAlertController.Style.alert)
        alertControl.addTextField { (textTF) in
            textTF.placeholder = "Please enter the number of copies".localized
        }
        
        let cancel = UIAlertAction(title: "Cancel".localized, style: UIAlertAction.Style.cancel, handler: nil)
        let strConfirm = UIAlertAction(title: "Text mode".localized, style: UIAlertAction.Style.default) { [weak alertControl](_) in
            guard let alert = alertControl else { return }
            guard let position = alert.textFields?.first, var temp = position.text else { return }
            if temp.isEmpty {
                temp = "1"
            }
            
            if let resultInt = Int(temp) {
                
                guard let data = try? Data.init(contentsOf: url) else {
                    SVProgressHUD.showError(withStatus: "Get File failed".localized)
                    return
                }
                guard let utf_string = PTEncode.decodeString(with: data) else {
                    SVProgressHUD.showInfo(withStatus: "The file is not supported".localized)
                    return
                }
                guard let fileData = PTEncode.encodeData(with: utf_string) else {
                    SVProgressHUD.showInfo(withStatus: "The file is not supported".localized)
                    return
                }
                var sendData = Data()
                for _ in 0..<resultInt {
                    sendData.append(fileData)
                }
                PDManagerSendData.shared.sendCommandData(sendData)
            }else {
                SVProgressHUD.showError(withStatus: "Incorrect input, please re-enter".localized)
            }
        }
        
        let hexConfirm = UIAlertAction(title: "Hex mode".localized, style: UIAlertAction.Style.default) { [weak alertControl](_) in
            guard let alert = alertControl else { return }
            guard let position = alert.textFields?.first, var temp = position.text else { return }
            if temp.isEmpty {
                temp = "1"
            }
            
            if let resultInt = Int(temp) {
                
                guard let data = try? Data.init(contentsOf: url) else {
                    SVProgressHUD.showError(withStatus: "Get File failed".localized)
                    return
                }
                guard let utf_string = PTEncode.decodeString(with: data) else {
                    SVProgressHUD.showInfo(withStatus: "The file is not supported".localized)
                    return
                }
                guard let fileData = utf_string.hexadecimalStringToData() else {
                    SVProgressHUD.showInfo(withStatus: "Irregular HEX data".localized)
                    return
                }
                var sendData = Data()
                for _ in 0..<resultInt {
                    sendData.append(fileData)
                }
                PDManagerSendData.shared.sendCommandData(sendData)
            }else {
                SVProgressHUD.showError(withStatus: "Incorrect input, please re-enter".localized)
            }
        }
        
        alertControl.addAction(strConfirm)
        alertControl.addAction(hexConfirm)
        alertControl.addAction(cancel)
        self.present(alertControl, animated: true, completion: nil)
    }
    
    func updatePrinterFirmware(_ url: URL) {
        UIAlertController.showActionAlert("Tips".localized, message: "Are you sure to upgrade the printer firmware?".localized, confirm: "OK".localized) { (_) in
            
            guard let deviceData = try? Data.init(contentsOf: url) else {
                SVProgressHUD.showError(withStatus: "Get File failed".localized)
                return
            }
            PDManagerSendData.shared.updateFirmware(deviceData)
        }
    }
    
    func selectPDFPrint(_ url: URL, _ cmdType: PTCommandType) {
        guard let _ = try? Data.init(contentsOf: url) else {
            SVProgressHUD.showError(withStatus: "Get File failed".localized)
            return
        }
        let ext = url.pathExtension.lowercased()
        if ext != "pdf" {
            SVProgressHUD.showInfo(withStatus: "The file is not supported".localized)
            return
        }
        if let navi = navigationController {
            let pdf = PDFPreviewViewController.init(with: url, cmdType: cmdType)
            navi.pushViewController(pdf, animated: true)
        }
       
    }
}
