//
//  ExampleTool.swift
//  Tool
//
//  Created by ldc on 2020/10/22.
//

import UIKit
import CommonCrypto
import QuickLook
import SVProgressHUD
import CoreServices
import AVFoundation
import Foundation

public extension Data {
    
    var md5: String { return encryptionString(type: .md5) }
    
    var sha1: String { return encryptionString(type: .sha1) }
    
    var sha256: String { encryptionString(type: .sha256) }
    
    private enum Encryption {
        case md5, sha1, sha256
    }
    
    private func encryptionString(type: Encryption) -> String {
        
        var length: Int
        switch type {
        case .md5:
            length = Int(CC_MD5_DIGEST_LENGTH)
        case .sha1:
            length = Int(CC_SHA1_DIGEST_LENGTH)
        case .sha256:
            length = Int(CC_SHA256_DIGEST_LENGTH)
        }
        let digest = UnsafeMutablePointer<UInt8>.allocate(capacity: length)
        switch type {
        case .md5:
            _ = CC_MD5((self as NSData).bytes, CC_LONG.init(self.count), digest)
        case .sha1:
            _ = CC_SHA1((self as NSData).bytes, CC_LONG.init(self.count), digest)
        case .sha256:
            _ = CC_SHA256((self as NSData).bytes, CC_LONG.init(self.count), digest)
        }
        let digestData = Data.init(bytes: digest, count: length)
        return digestData.hex.joined()
    }
}

public extension Data {
    
    static func create<T>(_ value: T, as type: T.Type) -> Data {
        
        let pointer = UnsafeMutablePointer<T>.allocate(capacity: 1)
        pointer.pointee = value
        let data = Data.init(bytes: UnsafeRawPointer(pointer), count: MemoryLayout<T>.stride)
        pointer.deallocate()
        return data
    }
    
    mutating func append<T>(_ value: T, as type: T.Type) -> Void {
        
        let pointer = UnsafeMutablePointer<T>.allocate(capacity: 1)
        pointer.pointee = value
        let raw = pointer.withMemoryRebound(to: UInt8.self, capacity: 1) { $0 }
        append(raw, count: MemoryLayout<T>.stride)
        pointer.deallocate()
    }
    
    mutating func storeBytes<T>(_ value: T, toByteOffset offset: Int = 0, as type: T.Type) {
        
        let pointer = UnsafeMutablePointer<T>.allocate(capacity: 1)
        pointer.pointee = value
        let size = MemoryLayout<T>.stride
        replaceSubrange(offset..<offset + size, with: UnsafePointer(pointer), count: size)
        pointer.deallocate()
    }
    
    func readValue<T>(offset: Int = 0, as type: T.Type) -> T? {
        
        guard offset >= 0 else { return nil }
        let size = MemoryLayout<T>.stride
        if offset + size > count {
            return nil
        }
        let bytes = (self as NSData).bytes + offset
        let pointer = bytes.bindMemory(to: type, capacity: 1)
        return pointer.pointee
    }
}

// MARK: CmdSample

struct CmdSample {
    var title = ""
    var detailTitle = ""
    var closure: (() -> Void)?
}

protocol CmdSampleGenerator {
    
    var samples: [CmdSample] { set get }
    
    var messageListener: ((String) -> Void)? { set get }
}

extension UIAlertController {
    
    struct Key {
        static var textDidChangeClosure = 0
    }
    
    @objc func textFieldTextDidChange() {
        
        textFieldTextDidChangeClosure?(textFields, actions)
    }
    
    var textFieldTextDidChangeClosure: (([UITextField]?, [UIAlertAction]) -> Void)? {
        
        set {
            objc_setAssociatedObject(self, &Key.textDidChangeClosure, newValue, .OBJC_ASSOCIATION_COPY_NONATOMIC)
        }
        
        get {
            return objc_getAssociatedObject(self, &Key.textDidChangeClosure) as? ([UITextField]?, [UIAlertAction]) -> Void
        }
    }
}

// MARK: UIView

public extension UIView {
    
    var viewController: UIViewController? {
        
        var result: UIViewController?
        var responder: UIResponder? = self
        while true  {
            if let _ = responder {
                if let temp = responder as? UIViewController {
                    result = temp
                    break
                }else {
                    responder = responder?.next
                }
            }else {
                break
            }
        }
        return result
    }
}

// MARK: Timer

public extension Timer {
    
    private struct Key {
        static var block = 0
        static var deadline = 0
    }
    
    private var closure: ((Timer) -> Void)? {
        
        set {
            objc_setAssociatedObject(self, &Key.block, newValue, .OBJC_ASSOCIATION_COPY_NONATOMIC)
        }
        
        get {
            return objc_getAssociatedObject(self, &Key.block) as? (Timer) -> Void
        }
    }
    
    var deadline: TimeInterval {
        
        set {
            objc_setAssociatedObject(self, &Key.deadline, newValue, .OBJC_ASSOCIATION_RETAIN_NONATOMIC)
        }
        
        get {
            return objc_getAssociatedObject(self, &Key.deadline) as? TimeInterval ?? 0
        }
    }
    
    @objc static func timerClosure(timer: Timer) {
        
        if let closure = timer.closure {
            timer.deadline -= timer.timeInterval
            if timer.isValid && timer.deadline <= 0 {
                timer.bk_invalidate()
            }
            closure(timer)
        }else {
            timer.bk_invalidate()
        }
    }
    
    @discardableResult
    static func timer(_ interval: TimeInterval, repeatCount: Int = 1, block: @escaping (Timer) -> Void) -> Timer {
        
        let timer = scheduledTimer(timeInterval: interval, target: self, selector: #selector(self.timerClosure(timer:)), userInfo: nil, repeats: true)
        timer.deadline = interval*TimeInterval(repeatCount)
        timer.closure = block
        return timer
    }
    
    func bk_invalidate() -> Void {
        
        closure = nil
        deadline = 0
        invalidate()
    }
}

// MARK: FilePreviewViewController
class FilePreviewViewController: QLPreviewController {
    
    let url: URL
    
    init(_ url: URL) {
        self.url = url
        super.init(nibName: nil, bundle: nil)
        dataSource = self
        delegate = self
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
}

extension FilePreviewViewController: QLPreviewControllerDataSource, QLPreviewControllerDelegate {
    func numberOfPreviewItems(in controller: QLPreviewController) -> Int {
        1
    }
    
    func previewController(_ controller: QLPreviewController, previewItemAt index: Int) -> QLPreviewItem {
        return url as QLPreviewItem
    }
}

// MARK: HDocumentPickerViewController

class HDocumentPickerViewController: UIDocumentPickerViewController {
    
    var didPickDocumentClosure: ((URL?) -> Void)?
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        view.backgroundColor = UIColor.white
        modalTransitionStyle = .coverVertical
        self.delegate = self
    }
}

extension HDocumentPickerViewController : UIDocumentPickerDelegate {
    
    func documentPickerWasCancelled(_ controller: UIDocumentPickerViewController) {
        print("documentPickerWasCancelled")
    }
    
    func documentPicker(_ controller: UIDocumentPickerViewController, didPickDocumentsAt urls: [URL]) {
        
        guard let url = urls.first else {
            didPickDocumentClosure?(nil)
            return
        }
        didPickDocument(at: url)
    }
    
    func documentPicker(_ controller: UIDocumentPickerViewController, didPickDocumentAt url: URL) {
        
        didPickDocument(at: url)
    }
    
    func didPickDocument(at url: URL) -> Void {
        
        didPickDocumentClosure?(url)
    }
}

// MARK: UIViewController

extension UIViewController {
    
    func presentInputAlertController(msg: String, defaultText: String? = nil, inputDesc: String, closure: @escaping ((String) -> Void)) -> Void {
        
        let temp = UIAlertController(title: "Tips", message: msg, preferredStyle: .alert)
        temp.addTextField {
            $0.placeholder = inputDesc
            $0.keyboardType = .numbersAndPunctuation
            $0.text = defaultText
        }
        let confirm = UIAlertAction(title: "OK", style: .default) { [weak temp] (_) in
            
            guard let textField = temp?.textFields?[0] else { return }
            guard let text = textField.text, !text.isEmpty else { return }
            closure(text)
        }
        let cancel = UIAlertAction(title: "Cancel", style: .cancel, handler: nil)
        temp.addAction(confirm)
        temp.addAction(cancel)
        present(temp, animated: true, completion: nil)
    }
}

extension UIViewController {
    
    func stringInput(_ title: String, textFieldSetter:((UITextField) -> Void)? = nil, doneEnable: Bool = false, completeClosure: @escaping ((String) -> Void)) -> Void {
        
        let temp = UIAlertController.init(title: title, message: nil, preferredStyle: .alert)
        var action = UIAlertAction.init(title: "取消", style: .cancel, handler: nil)
        temp.addAction(action)
        action = UIAlertAction.init(title: "确定", style: .default, handler: { [weak temp] (_) in
            guard let temp = temp, let textFields = temp.textFields, textFields.count > 0 else { return }
            guard let value = textFields[0].text, !value.isEmpty else { return }
            completeClosure(value)
        })
        action.isEnabled = doneEnable
        temp.addAction(action)
        temp.addTextField { [weak temp] in
            guard let temp = temp else { return }
            textFieldSetter?($0)
            NotificationCenter.default.addObserver(temp, selector: #selector(UIAlertController.textFieldTextDidChange), name: UITextField.textDidChangeNotification, object: $0)
        }
        temp.textFieldTextDidChangeClosure = {
            guard let textFields = $0, textFields.count > 0 else { return }
            guard let ssid = textFields[0].text else { return }
            if ssid.isEmpty {
                $1[1].isEnabled = false
            }else {
                $1[1].isEnabled = true
            }
        }
        self.present(temp, animated: true, completion: nil)
    }
    
    func valuePick<T>(_ values: [T], closure: @escaping (T) -> Void) -> Void {
        
        var actions = [UIAlertAction]()
        var action: UIAlertAction
        
        for i in values {
            action = UIAlertAction(title: "\(i)", style: .default, handler: { action in
                closure(i)
            })
            actions.append(action)
        }
        
        action = UIAlertAction.init(title: "取消", style: .cancel, handler: nil)
        actions.append(action)
        
        self.bk_presentAlertController(
            title: nil,
            message: "选择操作",
            preferredStyle: .actionSheet,
            actions: actions
        )
    }
    
    func filePick(_ extensionName: String, closure: @escaping (URL, Data) -> Void) -> Void {
        
        var temp: HDocumentPickerViewController
        if #available(iOS 14, *) {
            temp = HDocumentPickerViewController.init(
                forOpeningContentTypes: [
//                    UTType.init(tag: extensionName, tagClass: UTTagClass.filenameExtension, conformingTo: UTType.data)!
                    UTType.data,
                ], asCopy: true)
        }else {
            temp = HDocumentPickerViewController.init(documentTypes: [kUTTypeData as String], in: .import)
        }
        temp.modalPresentationStyle = .fullScreen
        temp.didPickDocumentClosure = {
            guard let url = $0 else {
                SVProgressHUD.showError(withStatus: "获取文件失败")
                return
            }
            DispatchQueue.global().async {
                do {
                    let data = try Data(contentsOf: url)
                    DispatchQueue.main.async {
                        closure(url, data)
                    }
                }catch {
                    DispatchQueue.main.async {
                        self.bk_presentWarningAlertController(title: "", message: error.localizedDescription, style: .default)
                    }
                }
            }
        }
        self.present(temp, animated: true, completion: nil)
    }
}

// MARK: UIImage

extension UIImage {
    
    /// 将图片自适应绘制到画布
    ///
    /// - Parameter size: 大小
    /// - Returns: 修改后图片
    func fit(size: CGSize) -> UIImage {
        
        UIGraphicsBeginImageContextWithOptions(size, true, 1)
        let scale = min(size.width / self.size.width, size.height / self.size.height)
        let drawSize = CGSize(width: floor(self.size.width * scale), height: floor(self.size.height * scale))
        draw(in: CGRect.init(
            x: floor((size.width - drawSize.width) / 2),
            y: floor((size.height - drawSize.height) / 2),
            width: drawSize.width,
            height: drawSize.height
        ))
        let result = UIGraphicsGetImageFromCurrentImageContext()!
        UIGraphicsEndImageContext()
        return result
    }
}

extension UserDefaults {
    
    func setDefaultIp(_ ip: String) -> Void {
        setValue(ip, forKey: "target_ip")
        synchronize()
    }
    
    func getDefaultIp() -> String? {
        return value(forKey: "target_ip") as? String
    }
}

public struct AppInfo {
    
    public static var version: String = { Bundle.main.infoDictionary?["CFBundleShortVersionString"] as? String ?? "" }()
    
    public static var bundleId: String = { Bundle.main.infoDictionary?["CFBundleIdentifier"] as? String ?? "" }()
    
    public static var displayName: String = {
        Bundle.main.localizedInfoDictionary?["CFBundleDisplayName"] as? String ?? Bundle.main.infoDictionary?["CFBundleDisplayName"] as? String ?? ""
    }()
    
    public static var build: String = { Bundle.main.infoDictionary?["CFBundleVersion"] as? String ?? "0" }()
}

public struct SandBox {
    
    public static let documents = NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask, true)[0]
    
    public static let cache = NSSearchPathForDirectoriesInDomains(.cachesDirectory, .userDomainMask, true)[0]
}

extension Result {
    
    static func from(success: Success?, failure: Failure?) -> Result<Success,Failure> {
        if let failure = failure {
            return .failure(failure)
        }
        return .success(success!)
    }
}

extension Result where Success == Void {
    
    static func from(failure: Failure?) -> Result<Void,Failure> {
        if let failure = failure {
            return .failure(failure)
        }
        return .success(())
    }
}
